'use client';

import React, { createContext, useContext, useState, ReactNode, useEffect, useMemo } from 'react';

interface LanguageContextType {
  language: string;
  setLanguage: (lang: string) => void;
  isRTL: boolean;
  defaultLang: string;
}

const LanguageContext = createContext<LanguageContextType>({
  language: 'en',
  setLanguage: () => {},
  isRTL: false,
  defaultLang: 'en',
});

export function LanguageProvider({ children }: { children: ReactNode }) {
  const defaultLang = process.env.NEXT_PUBLIC_DEFAULT_LANG || 'en';
  const [language, setLanguage] = useState(defaultLang);

  // استرجاع من localStorage
  useEffect(() => {
    const saved = localStorage.getItem('language');
    if (saved) setLanguage(saved);
  }, []);

  // حفظ كل مرة
  useEffect(() => {
    localStorage.setItem('language', language);
  }, [language]);

  const isRTL = useMemo(() => ['ar', 'he', 'fa', 'ur'].includes(language), [language]);

  return (
    <LanguageContext.Provider value={{ language, setLanguage, isRTL, defaultLang }}>
      {children}
    </LanguageContext.Provider>
  );
}

export function useLanguage() {
  return useContext(LanguageContext);
}
