"use client";

import { useState, useEffect } from "react";
import Link from "next/link";
import Image from "next/image";

interface LogoData {
  logo: string | null;
  logo_light: string | null;
  width: number;
  height: number;
}

export default function Logo() {
  const [loading, setLoading] = useState(true);
  const [logoData, setLogoData] = useState<LogoData | null>(null);

  useEffect(() => {
    const fetchLogo = async () => {
      try {
        const res = await fetch(
          `${process.env.NEXT_PUBLIC_WORDPRESS_URL}/wp-json/tewido/v1/logo`,
          { cache: "no-store" }
        );
        const data = await res.json();
        setLogoData({
          logo: data.logo,
          logo_light: data.logo_light,
          width: data.width || 150,
          height: data.height || 50,
        });
      } catch (error) {
        console.error("Error fetching logo:", error);
        setLogoData(null);
      } finally {
        setLoading(false);
      }
    };

    fetchLogo();
  }, []);

  return (
    <Link href="/" className="flex items-center">
      {loading ? (
        <div className="h-12 w-12 rounded-full bg-gray-200 dark:bg-gray-700 animate-pulse" />
      ) : logoData && (logoData.logo || logoData.logo_light) ? (
        <>
          {/* ✅ Main Logo (Light Mode Only) */}
          {logoData.logo && (
            <Image
              src={logoData.logo}
              alt="Site Logo"
              width={logoData.width}
              height={logoData.height}
              className="h-8 w-auto object-contain block dark:hidden"
              priority
            />
          )}

          {/* ✅ Light Logo (Dark Mode Only) */}
          {logoData.logo_light && (
            <Image
              src={logoData.logo_light}
              alt="Site Logo Light"
              width={logoData.width}
              height={logoData.height}
              className="h-8 w-auto object-contain hidden dark:block"
              priority
            />
          )}
        </>
      ) : (
        <span className="text-gray-500 dark:text-gray-300 font-bold">LOGO</span>
      )}
    </Link>
  );
}
