"use client";

import { useState, useEffect } from 'react';
import { usePathname } from 'next/navigation';
import { useLanguage } from '@/contexts/LanguageContext';
import Link from 'next/link';

interface BreadcrumbProps {
  className?: string;
}

const Breadcrumb: React.FC<BreadcrumbProps> = ({ className = "" }) => {
  const [isClient, setIsClient] = useState(false);
  const pathname = usePathname();
  const { language } = useLanguage();

  // Set client-side flag
  useEffect(() => {
    setIsClient(true);
  }, []);

  // If not client-side, return null to avoid rendering
  if (!isClient) return null;

  // Split the path into segments
  const pathSegments = pathname.split('/').filter(segment => segment.length > 0);

  // Helper function to get display name for segments
  const getSegmentDisplayName = (segment: string, index: number) => {
    // Customize display names for specific segments
    const segmentMap: { [key: string]: string } = {
      'posts': language === 'ar' ? 'المقالات' : 'Posts',
      'category': language === 'ar' ? 'التصنيف' : 'Category',
      'page': language === 'ar' ? 'الصفحات' : 'Pages',
    };

    return segmentMap[segment] || segment.charAt(0).toUpperCase() + segment.slice(1);
  };

  // Helper function to get correct href for each segment
  const getSegmentHref = (segment: string, index: number) => {
    const currentPath = pathSegments.slice(0, index + 1);

    // Special handling for category segment
    if (segment === 'category' && pathSegments[index - 1] === 'posts') {
      return '/posts/category'; // Link to categories index
    }

    // For all other segments, build the path normally
    return `/${currentPath.join('/')}`;
  };

  // Helper function to determine if a segment should be clickable
  const isSegmentClickable = (segment: string, index: number, isLast: boolean) => {
    // Don't make the last segment clickable (current page)
    if (isLast) return false;

    // Special case: if we're on a specific category page (/posts/category/slug)
    // make the "category" segment clickable to go to categories index
    if (segment === 'category' && pathSegments[index - 1] === 'posts') {
      return true;
    }

    // Make other segments clickable if they have valid routes
    if (segment === 'posts') return true;

    return false;
  };

  return (
    <div className={`site-breadcrumb bg-gray-100 dark:bg-gray-700/20 py-3 ${className}`}>
      <nav className="container" aria-label="Breadcrumb">
        <ol className="flex space-x-2 text-sm text-gray-600 dark:text-gray-300">
          <li>
            <Link href={`/${language === 'ar' ? 'ar' : ''}`} className="text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-500">
              {language === 'ar' ? 'الرئيسية' : 'Home'}
            </Link>
          </li>
          {pathSegments.map((segment, index) => {
            const isLast = index === pathSegments.length - 1;
            const displayName = getSegmentDisplayName(segment, index);
            const href = getSegmentHref(segment, index);
            const clickable = isSegmentClickable(segment, index, isLast);

            return (
              <li key={index} className="flex items-center">
                <span className="mx-2">/</span>
                {isLast || !clickable ? (
                  <span className="text-gray-500 dark:text-gray-400">{displayName}</span>
                ) : (
                  <Link href={href} className="text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-500">
                    {displayName}
                  </Link>
                )}
              </li>
            );
          })}
        </ol>
      </nav>
    </div>
  );
};

export default Breadcrumb;