'use client';

import { useEffect, useState } from "react";
import wordpressAPI from "@/lib/wordpress";
import { useLanguage } from "@/contexts/LanguageContext";
import Icon from "@/components/icons/Icon";
import Link from "next/link";

// 🖼️ Icons
import {
  PiFacebookLogoLight,
  PiTwitterLogoLight,
  PiInstagramLogoLight,
  PiYoutubeLogoLight,
  PiLinkedinLogoLight,
  PiWhatsappLogoLight,
  PiSnapchatLogoLight,
  PiTiktokLogoLight,
  PiInfoLight,
} from "react-icons/pi";

interface ContactInfoProps {
  showInfo?: boolean;
  showSocial?: boolean;
  showMap?: boolean;
  fields?: string[];
}

export default function ContactInfo({
  showInfo = true,
  showSocial = true,
  showMap = true,
  fields,
}: ContactInfoProps) {
  const { language } = useLanguage();
  const [data, setData] = useState<any | null>(null);
  const [loading, setLoading] = useState(true);

  const infoIconClasses = "w-5 h-5 text-current hover:text-primary-500";
  const socialIconClasses = "w-6 h-6 text-current hover:text-primary-500";

  useEffect(() => {
    const fetchData = async () => {
      try {
        const res = await wordpressAPI.getContactInfo();
        setData(res);
      } catch (error) {
        console.error("Error fetching contact info:", error);
        setData(null);
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, [language]);

  const shouldShow = (key: string) => !fields || fields.includes(key);

  // ==================
  // 🔹 Skeleton loaders منفصلة
  // ==================
  const InfoSkeleton = () => (
    <div className="space-y-2">
      <div className="h-4 w-40 bg-gray-200 dark:bg-gray-700 animate-pulse rounded" />
      <div className="h-4 w-56 bg-gray-200 dark:bg-gray-700 animate-pulse rounded" />
      <div className="h-4 w-32 bg-gray-200 dark:bg-gray-700 animate-pulse rounded" />
    </div>
  );

  const SocialSkeleton = () => (
    <div className="flex gap-3">
      {Array.from({ length: 5 }).map((_, i) => (
        <div
          key={i}
          className="w-6 h-6 bg-gray-200 dark:bg-gray-700 animate-pulse rounded-full"
        />
      ))}
    </div>
  );

  const MapSkeleton = () => (
    <div className="w-full h-[250px] bg-gray-200 dark:bg-gray-700 animate-pulse rounded-md" />
  );

  if (!data && !loading) {
    return (
      <p className="flex items-center gap-2 text-gray-400 dark:text-gray-500">
        <PiInfoLight className={infoIconClasses} />
        {language === "ar"
          ? "لا توجد بيانات تواصل متاحة حالياً."
          : "No contact information available at the moment."}
      </p>
    );
  }

  return (
    <div className="contact-info space-y-6">
      {/* ✅ Info */}
      {showInfo &&
        (loading ? (
          <InfoSkeleton />
        ) : (
          <div className="info space-y-2">
            {shouldShow("phone") && data?.phone && (
              <p className="flex items-center gap-2">
                <Icon name="phone" size={18} />
                <a
                  href={`tel:${data.phone}`}
                  className="hover:underline"
                  aria-label={`Call ${data.phone}`}
                >
                  {data.phone}
                </a>
              </p>
            )}
            {shouldShow("email") && data?.email && (
              <p className="flex items-center gap-2">
                <Icon name="email" size={18} />
                <a
                  href={`mailto:${data.email}`}
                  className="hover:underline"
                  aria-label={`Send email to ${data.email}`}
                >
                  {data.email}
                </a>
              </p>
            )}
            {shouldShow("address_ltr") &&
              language === "en" &&
              data?.address_ltr && (
                <p className="flex items-center gap-2">
                  <Icon name="map" size={18} />
                  <span>{data.address_ltr}</span>
                </p>
              )}
            {shouldShow("address_rtl") &&
              language === "ar" &&
              data?.address_rtl && (
                <p className="flex items-center gap-2">
                  <Icon name="map" size={18} />
                  <span>{data.address_rtl}</span>
                </p>
              )}
            {shouldShow("working_hours") && data?.working_hours && (
              <p className="flex items-center gap-2">
                <Icon name="clock" size={18} />
                <span>{data.working_hours}</span>
              </p>
            )}
          </div>
        ))}

      {/* ✅ Social */}
      {showSocial &&
        (loading ? (
          <SocialSkeleton />
        ) : (
          <div className="social flex gap-3">
            {data?.facebook_url && (
              <Link
                href={data.facebook_url}
                target="_blank"
                aria-label="Visit our Facebook page"
              >
                <PiFacebookLogoLight className={socialIconClasses} />
              </Link>
            )}
            {data?.twitter_url && (
              <Link
                href={data.twitter_url}
                target="_blank"
                aria-label="Visit our Twitter page"
              >
                <PiTwitterLogoLight className={socialIconClasses} />
              </Link>
            )}
            {data?.instagram_url && (
              <Link
                href={data.instagram_url}
                target="_blank"
                aria-label="Visit our Instagram page"
              >
                <PiInstagramLogoLight className={socialIconClasses} />
              </Link>
            )}
            {data?.youtube_url && (
              <Link
                href={data.youtube_url}
                target="_blank"
                aria-label="Visit our YouTube channel"
              >
                <PiYoutubeLogoLight className={socialIconClasses} />
              </Link>
            )}
            {data?.linkedin_url && (
              <Link
                href={data.linkedin_url}
                target="_blank"
                aria-label="Visit our LinkedIn profile"
              >
                <PiLinkedinLogoLight className={socialIconClasses} />
              </Link>
            )}
            {data?.whatsapp_url && (
              <Link
                href={`https://wa.me/${data.whatsapp_url.replace(/\D/g, "")}`}
                target="_blank"
                aria-label="Chat with us on WhatsApp"
              >
                <PiWhatsappLogoLight className={socialIconClasses} />
              </Link>
            )}
            {data?.snapchat_url && (
              <Link
                href={data.snapchat_url}
                target="_blank"
                aria-label="Visit our Snapchat"
              >
                <PiSnapchatLogoLight className={socialIconClasses} />
              </Link>
            )}
            {data?.tiktok_url && (
              <Link
                href={data.tiktok_url}
                target="_blank"
                aria-label="Visit our TikTok"
              >
                <PiTiktokLogoLight className={socialIconClasses} />
              </Link>
            )}
          </div>
        ))}

      {/* ✅ Map */}
      {showMap &&
        (loading ? (
          <MapSkeleton />
        ) : (
          <div className="map">
            {data?.google_map ? (
              <iframe
                src={data.google_map}
                width="100%"
                height="400"
                style={{ border: 0 }}
                loading="lazy"
                allowFullScreen
                className="rounded-md"
              />
            ) : (
              <MapSkeleton />
            )}
          </div>
        ))}
    </div>
  );
}
