"use client";

import { useEffect, useState } from "react";
import Link from "next/link";
import wordpressAPI from "@/lib/wordpress";
import SectionHeader from "@/components/common/SectionHeader";
import { useLanguage } from "@/contexts/LanguageContext";

interface Category {
  id: number;
  name: string;
  slug: string;
  description: string;
  count: number;
}

export default function CategoriesIndex() {
  const { language } = useLanguage();
  const [categories, setCategories] = useState<Category[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchCategories = async () => {
      try {
        setLoading(true);
        const data = await wordpressAPI.getCategories(language);
        setCategories(data || []);
      } catch (error) {
        console.error("Error loading categories:", error);
        setCategories([]);
      } finally {
        setLoading(false);
      }
    };

    fetchCategories();
  }, [language]);

  return (
    <div className="te-categories my-12">
      <div className="container">
        
        {/* Section Header */}
        <SectionHeader
          title={language === "ar" ? "التصنيفات" : "Categories"}
          subtitle={language === "ar" ? "تصفح جميع تصنيفات المقالات" : "Browse all post categories"}
          level="h2"
          marginBottom="8"
        />

        {/* Categories Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {loading
            ? Array.from({ length: 6 }).map((_, i) => (
                <div
                  key={i}
                  className="animate-pulse bg-white dark:bg-gray-800 rounded-lg shadow-md p-6"
                >
                  <div className="h-6 bg-gray-200 dark:bg-gray-700 rounded w-3/4 mb-3" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-full mb-2" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-1/2" />
                </div>
              ))
            : categories.length > 0
            ? categories.map((category) => (
                <Link
                  key={category.id}
                  href={`/posts/category/${category.slug}`}
                  className="block bg-white dark:bg-gray-800 rounded-lg shadow-md p-6 hover:shadow-lg transition-shadow duration-200 group"
                >
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2 group-hover:text-primary-600 dark:group-hover:text-primary-400">
                    {category.name}
                  </h3>
                  
                  {category.description && (
                    <p className="text-sm text-gray-600 dark:text-gray-400 mb-3 line-clamp-2">
                      {category.description}
                    </p>
                  )}
                  
                  <div className="text-xs text-gray-500 dark:text-gray-500">
                    {language === "ar" 
                      ? `${category.count} منشور`
                      : `${category.count} post${category.count !== 1 ? 's' : ''}`
                    }
                  </div>
                </Link>
              ))
            : (
                <div className="col-span-full text-center py-12">
                  <p className="text-gray-500 dark:text-gray-400">
                    {language === "ar"
                      ? "لا توجد تصنيفات متاحة"
                      : "No categories available"
                    }
                  </p>
                </div>
              )}
        </div>
      </div>
    </div>
  );
}