// File: app/[your-post-type]/category/page.tsx
// Replace [your-post-type] with your actual custom post type name

"use client";

import { useEffect, useState } from "react";
import Link from "next/link";
import wordpressAPI from "@/lib/wordpress";
import SectionHeader from "@/components/common/SectionHeader";
import Button from '@/components/ui/Button';
import { useLanguage } from "@/contexts/LanguageContext";

interface CustomTaxonomyTerm {
  id: number;
  name: string;
  slug: string;
  description: string;
  count: number;
}

// 🚨 IMPORTANT: Update these constants for your custom post type
const POST_TYPE = 'portfolio'; // Change to your custom post type name (e.g., 'services', 'portfolio', etc.)
const TAXONOMY = 'portfolios'; // Change to your custom taxonomy name
const POST_TYPE_LABEL = 'Portfolio'; // Human readable label for your post type
const TAXONOMY_LABEL = 'Portfolio Categories'; // Human readable label for your taxonomy

export default function CustomPostTypeCategoriesIndex() {
  const { language } = useLanguage();
  const [categories, setCategories] = useState<CustomTaxonomyTerm[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchCategories = async () => {
      try {
        setLoading(true);
        const data = await wordpressAPI.getCustomTaxonomyTerms(TAXONOMY, language);
        setCategories(data || []);
      } catch (error) {
        console.error("Error loading categories:", error);
        setCategories([]);
      } finally {
        setLoading(false);
      }
    };

    fetchCategories();
  }, [language]);

  return (
    <div className="te-categories my-12">
      <div className="container">

        <div className="mb-8">
          <div className="grid grid-cols-[1fr_auto] gap-6 items-end">
            <div>
              {/* Section Header */}
              <SectionHeader
                title={
                  language === "ar"
                    ? `تصنيفات ${POST_TYPE_LABEL}`
                    : TAXONOMY_LABEL
                }
                subtitle={
                  language === "ar"
                    ? `تصفح جميع تصنيفات ${POST_TYPE_LABEL.toLowerCase()}`
                    : `Browse all ${POST_TYPE_LABEL.toLowerCase()} categories`
                }
                level="h2"
                marginBottom="0"
                textAlign="start"
              />
            </div>

            <div className="w-auto">
              {/* Categories Stats */}
              {!loading && categories.length > 0 && (
                <div className="text-sm text-gray-600 dark:text-gray-400">
                  {language === "ar"
                    ? `${categories.length} تصنيف متاح`
                    : `${categories.length} categor${categories.length !== 1 ? 'ies' : 'y'} available`
                  }
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Categories Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {loading
            ? Array.from({ length: 6 }).map((_, i) => (
              <div
                key={i}
                className="animate-pulse bg-white dark:bg-gray-800 rounded-lg shadow-md p-6 border border-gray-200 dark:border-gray-700"
              >
                <div className="space-y-4">
                  <div className="h-6 bg-gray-200 dark:bg-gray-700 rounded w-3/4" />
                  <div className="space-y-2">
                    <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-full" />
                    <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-2/3" />
                  </div>
                  <div className="h-3 bg-gray-200 dark:bg-gray-700 rounded w-1/3" />
                </div>
              </div>
            ))
            : categories.length > 0
              ? categories.map((category) => (
                <Link
                  key={category.id}
                  href={`/${POST_TYPE}/category/${category.slug}`}
                  className="group block bg-white dark:bg-gray-800 rounded-lg shadow-md border border-gray-200 dark:border-gray-700 overflow-hidden hover:shadow-lg transition-all duration-200 hover:border-primary-200 dark:hover:border-primary-700"
                >
                  <div className="p-6">
                    {/* Category Name */}
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-3 group-hover:text-primary-600 dark:group-hover:text-primary-400 transition-colors duration-200">
                      {category.name}
                    </h3>

                    {/* Category Description */}
                    {category.description && (
                      <p className="text-sm text-gray-600 dark:text-gray-400 mb-4 line-clamp-3 leading-relaxed">
                        {category.description}
                      </p>
                    )}

                    {/* Category Stats */}
                    <div className="flex items-center justify-between">
                      <div className="flex items-center text-xs text-gray-500 dark:text-gray-500">
                        <svg
                          className="w-4 h-4 me-1"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                          />
                        </svg>
                        {language === "ar"
                          ? `${category.count} عنصر`
                          : `${category.count} item${category.count !== 1 ? 's' : ''}`
                        }
                      </div>

                      {/* Arrow Icon */}
                      <svg
                        className={`w-4 h-4 text-gray-400 group-hover:text-primary-500 transition-all duration-200 group-hover:translate-x-1 ${language === 'ar' ? 'rotate-180 group-hover:-translate-x-1' : ''
                          }`}
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M9 5l7 7-7 7"
                        />
                      </svg>
                    </div>
                  </div>

                  {/* Hover Effect Border */}
                  <div className="h-1 bg-gradient-to-r from-primary-500 to-secondary-500 transform scale-x-0 group-hover:scale-x-100 transition-transform duration-200 origin-left"></div>
                </Link>
              ))
              : (
                <div className="col-span-full">
                  <div className="text-center py-16">
                    <div className="mb-4">
                      <svg
                        className="mx-auto h-16 w-16 text-gray-300 dark:text-gray-600"
                        fill="none"
                        viewBox="0 0 24 24"
                        stroke="currentColor"
                        strokeWidth={1}
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"
                        />
                      </svg>
                    </div>
                    <h3 className="text-xl font-medium text-gray-900 dark:text-white mb-2">
                      {language === "ar"
                        ? "لا توجد تصنيفات"
                        : "No categories found"
                      }
                    </h3>
                    <p className="text-gray-500 dark:text-gray-400 max-w-sm mx-auto">
                      {language === "ar"
                        ? `لا توجد تصنيفات متاحة لـ ${POST_TYPE_LABEL.toLowerCase()} حالياً.`
                        : `No categories are available for ${POST_TYPE_LABEL.toLowerCase()} yet.`
                      }
                    </p>
                  </div>
                </div>
              )}
        </div>

        {/* Back to Main Link */}
        {!loading && (
          <div className="mt-12 text-center">

            <Button href={`/${POST_TYPE}`} variant="primary" size="md">
              {language === "ar"
                ? `العودة إلى جميع ${POST_TYPE_LABEL}`
                : `Back to all ${POST_TYPE_LABEL}`
              }
            </Button>
          </div>
        )}
      </div>
    </div>
  );
}