"use client";

import { useEffect, useState } from "react";

export default function DarkModeToggle() {
    const [isDark, setIsDark] = useState(false);

    // ✅ Check system & localStorage on mount
    useEffect(() => {
        const saved = localStorage.getItem("darkMode");
        const systemDark = window.matchMedia("(prefers-color-scheme: dark)").matches;
        const active = saved === "true" || (!saved && systemDark);

        setIsDark(active);
        document.documentElement.classList.toggle("dark", active);

        // ✅ Listen for system changes
        const media = window.matchMedia("(prefers-color-scheme: dark)");
        const systemListener = (e: MediaQueryListEvent) => {
            if (localStorage.getItem("darkMode") === null) {
                setIsDark(e.matches);
                document.documentElement.classList.toggle("dark", e.matches);
            }
        };
        media.addEventListener("change", systemListener);
        return () => media.removeEventListener("change", systemListener);
    }, []);

    // ✅ Apply theme + save
    const applyTheme = (dark: boolean) => {
        setIsDark(dark);
        document.documentElement.classList.toggle("dark", dark);
        localStorage.setItem("darkMode", dark ? "true" : "false");
    };

    return (
        <button
            onClick={() => applyTheme(!isDark)}
            className="w-10 h-10 flex items-center justify-center text-gray-900 dark:text-white
    rounded-md 
    border border-gray-300 dark:border-gray-600 
    transition-transform duration-300 
    focus:outline-none focus:ring-0"
        >
            {isDark ? (
                // 🌙 Moon icon
                <svg
                    xmlns="http://www.w3.org/2000/svg"
                    width="20"
                    height="20"
                    viewBox="0 0 24 24"
                    fill="none"
                    stroke="currentColor"
                    strokeWidth="2"
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    className="lucide lucide-moon-icon w-5 h-5 animate-spin-slow"
                >
                    <path d="M20.985 12.486a9 9 0 1 1-9.473-9.472c.405-.022.617.46.402.803a6 6 0 0 0 8.268 8.268c.344-.215.825-.004.803.401" />
                </svg>
            ) : (
                // ☀️ Sun icon
                <svg
                    xmlns="http://www.w3.org/2000/svg"
                    width="20"
                    height="20"
                    viewBox="0 0 24 24"
                    fill="none"
                    stroke="currentColor"
                    strokeWidth="2"
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    className="lucide lucide-sun-medium-icon w-5 h-5 animate-spin-slow"
                >
                    <circle cx="12" cy="12" r="4" />
                    <path d="M12 3v1" />
                    <path d="M12 20v1" />
                    <path d="M3 12h1" />
                    <path d="M20 12h1" />
                    <path d="m18.364 5.636-.707.707" />
                    <path d="m6.343 17.657-.707.707" />
                    <path d="m5.636 5.636.707.707" />
                    <path d="m17.657 17.657.707.707" />
                </svg>
            )}
        </button>
    );
}
