'use client';

import { useEffect, useState } from 'react';
import Link from 'next/link';
import { useLanguage } from '@/contexts/LanguageContext';
import wordpressAPI from '@/lib/wordpress';

interface MenuItem {
  id: number;
  title: string;
  url: string;
  target?: string;
  parent: string | number;
  children?: MenuItem[];
}

interface MenuProps {
  mobile?: boolean;
  onLinkClick?: () => void;
}

export default function Menu({ mobile, onLinkClick }: MenuProps) {
  const [menuItems, setMenuItems] = useState<MenuItem[]>([]);
  const [loading, setLoading] = useState(true);
  const { language, defaultLang } = useLanguage();

  useEffect(() => {
    const fetchMenu = async () => {
      try {
        setLoading(true);
        const menus = await wordpressAPI.getMenus(language);
        const items = menus['header-menu'] || [];
        setMenuItems(buildTree(items));
      } finally {
        setLoading(false);
      }
    };
    fetchMenu();
  }, [language]);

  // ✅ turn flat array into nested tree
  const buildTree = (items: MenuItem[]) => {
    const map: Record<string | number, MenuItem> = {};
    const roots: MenuItem[] = [];

    items.forEach((item) => {
      map[item.id] = { ...item, children: [] };
    });

    items.forEach((item) => {
      if (item.parent && map[item.parent]) {
        map[item.parent].children!.push(map[item.id]);
      } else {
        roots.push(map[item.id]);
      }
    });

    return roots;
  };

  // ✅ normalize URL
  const formatUrl = (url: string) => {
    if (url.includes(process.env.NEXT_PUBLIC_WORDPRESS_URL!)) {
      url = url.replace(process.env.NEXT_PUBLIC_WORDPRESS_URL!, '');
    }
    url = url.replace(/^\/(ar|en)(?=\/|$)/, '');
    if (language !== defaultLang) {
      url = `/${language}${url}`;
    }
    return url || '/';
  };

  if (mobile) {
    return (
      <div className="divide-y divide-gray-200 dark:divide-gray-700 space-y-2">
        {loading
          ? Array.from({ length: 4 }).map((_, i) => (
            <div
              key={i}
              className="h-5 w-full bg-gray-200 dark:bg-gray-700 rounded-md animate-pulse"
            />
          ))
          : menuItems.map((item) => (
            <MobileMenuItem
              key={item.id}
              item={item}
              formatUrl={formatUrl}
              onLinkClick={onLinkClick}
            />
          ))}
      </div>
    );
  }

  return (
    <ul className="hidden md:flex items-center gap-6">
      {loading
        ? Array.from({ length: 4 }).map((_, i) => (
          <div
            key={i}
            className="h-5 w-20 bg-gray-200 dark:bg-gray-700 animate-pulse rounded"
          />
        ))
        : menuItems.map((item) => (
          <DesktopMenuItem key={item.id} item={item} formatUrl={formatUrl} />
        ))}
    </ul>
  );
}

/* ✅ Desktop item with hover submenu + arrow */
function DesktopMenuItem({
  item,
  formatUrl,
}: {
  item: MenuItem;
  formatUrl: (url: string) => string;
}) {
  return (
    <li className="relative group">
      <div className="flex items-center gap-1">
        <Link
          href={formatUrl(item.url)}
          target={item.target}
          className="text-sm text-gray-700 dark:text-gray-200 hover:text-primary-600 dark:hover:text-primary-400 font-semibold transition-colors"
        >
          {item.title}
        </Link>
        {item.children && item.children.length > 0 && (
          <svg
            className="w-4 h-4 text-gray-500 dark:text-gray-400 transition-transform duration-200 group-hover:rotate-180"
            fill="none"
            stroke="currentColor"
            strokeWidth="2"
            viewBox="0 0 24 24"
          >
            <path strokeLinecap="round" strokeLinejoin="round" d="M19 9l-7 7-7-7" />
          </svg>
        )}
      </div>

      {item.children && item.children.length > 0 && (
        <ul className="absolute start-0 mt-2 w-40 bg-white dark:bg-gray-800 shadow-lg rounded-md opacity-0 group-hover:opacity-100 group-hover:translate-y-0 transform translate-y-2 transition-all duration-200 z-50">
          {item.children.map((child) => (
            <li key={child.id}>
              <Link
                href={formatUrl(child.url)}
                className="block px-4 py-2 text-sm text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
              >
                {child.title}
              </Link>
            </li>
          ))}
        </ul>
      )}
    </li>
  );
}

/* ✅ Mobile item with arrow toggle + border */
function MobileMenuItem({
  item,
  formatUrl,
  onLinkClick,
}: {
  item: MenuItem;
  formatUrl: (url: string) => string;
  onLinkClick?: () => void;
}) {
  const [open, setOpen] = useState(false);

  return (
    <div className="pt-2">
      <div className="flex justify-between items-center">
        <Link
          href={formatUrl(item.url)}
          target={item.target}
          onClick={onLinkClick}
          className="block text-gray-700 dark:text-gray-200 hover:text-primary-600 dark:hover:text-primary-400 transition-colors"
        >
          {item.title}
        </Link>
        {item.children && item.children.length > 0 && (
          <button
            onClick={() => setOpen(!open)}
            className="text-gray-600 dark:text-gray-300 transition-transform duration-200"
          >
            <svg
              className={`w-4 h-4 transform transition-transform duration-200 ${open ? 'rotate-180' : ''
                }`}
              fill="none"
              stroke="currentColor"
              strokeWidth="2"
              viewBox="0 0 24 24"
            >
              <path strokeLinecap="round" strokeLinejoin="round" d="M19 9l-7 7-7-7" />
            </svg>
          </button>
        )}
      </div>

      {open && item.children && (
        <div className="mt-2 space-y-1">
          {item.children.map((child) => (
            <Link
              key={child.id}
              href={formatUrl(child.url)}
              onClick={onLinkClick}
              className="block py-1 text-sm text-gray-600 dark:text-gray-300 hover:text-primary-600 dark:hover:text-primary-400"
            >
              {child.title}
            </Link>
          ))}
        </div>
      )}
    </div>
  );
}
