import Link from "next/link";
import { useLanguage } from "@/contexts/LanguageContext";

interface PostCardProps {
    post: any; // Replace `any` with your Post type from wordpressAPI
}

export default function PostCard({ post }: PostCardProps) {
    const { language } = useLanguage();

    // WordPress featured image (if embedded)
    const featuredImage =
        post._embedded?.["wp:featuredmedia"]?.[0]?.source_url || null;
    const altText =
        post._embedded?.["wp:featuredmedia"]?.[0]?.alt_text || post.title.rendered;

    return (
        <article className="bg-white dark:bg-gray-800 rounded-lg shadow-sm transition-shadow overflow-hidden">
            {/* Thumbnail */}
            <div className="relative h-48 w-full">
                {!featuredImage ? (
                    // Placeholder when no image is available
                    <div className="placeholder bg-gray-200 dark:bg-gray-700 h-full flex items-center justify-center">
                        <span className="text-gray-500 dark:text-gray-300">
                            {language === "ar" ? "لا توجد صورة" : "No Image Available"}
                        </span>
                    </div>
                ) : (
                    <img
                        src={featuredImage}
                        alt={altText}
                        className="object-cover w-full h-full"
                    />
                )}
            </div>

            <div className="p-6">
                <h3 className="te-card-title mb-2">
                    <Link
                        href={`/posts/${post.slug}`}
                        className="text-lg font-semibold text-gray-900 dark:text-white hover:text-primary-700 mb-2 leading-tight"
                    >
                        <span
                            dangerouslySetInnerHTML={{ __html: post.title.rendered }}
                        />
                    </Link>
                </h3>

                <div
                    className="te-card-text text-sm text-gray-700 dark:text-gray-200 leading-relaxed mt-4 line-clamp-3"
                    dangerouslySetInnerHTML={{ __html: post.excerpt.rendered }}
                />

                <Link
                    href={`/posts/${post.slug}`}
                    className="inline-flex items-center gap-1 text-primary-700 dark:text-primary-400 hover:underline font-semibold mt-4 text-sm"
                >
                    {language === "ar" ? "اقرأ المزيد" : "Read More"}
                    <svg
                        xmlns="http://www.w3.org/2000/svg"
                        className={`w-4 h-4 ${language === "ar" ? "rotate-180" : ""}`}
                        fill="none"
                        viewBox="0 0 24 24"
                        stroke="currentColor"
                        strokeWidth={2}
                    >
                        <path strokeLinecap="round" strokeLinejoin="round" d="M9 5l7 7-7 7" />
                    </svg>
                </Link>
            </div>
        </article>
    );
}
