'use client';

import { useEffect, useState } from 'react';
import Link from 'next/link';
import { useLanguage } from '@/contexts/LanguageContext';
import ContactInfo from '@/components/customizer/ContactInfo';
import wordpressAPI from '@/lib/wordpress';

interface MenuItem {
  id: number;
  title: string;
  url: string;
  target?: string;
  parent: string | number;
}

interface FooterData {
  footer_about_en: string;
  footer_about_ar: string;
}

export default function Footer() {
  const [footerMenu1, setFooterMenu1] = useState<MenuItem[]>([]);
  const [footerMenu2, setFooterMenu2] = useState<MenuItem[]>([]);
  const [footerData, setFooterData] = useState<FooterData | null>(null);
  const [translations, setTranslations] = useState<any | null>(null);
  const { language } = useLanguage();

  // ✅ Fetch Menus
  useEffect(() => {
    const fetchMenus = async () => {
      const menus = await wordpressAPI.getMenus(language);
      setFooterMenu1(menus['footer-menu-1'] || []);
      setFooterMenu2(menus['footer-menu-2'] || []);
    };
    fetchMenus();
  }, [language]);

  // ✅ Fetch Footer Data
  useEffect(() => {
    const loadFooter = async () => {
      const data = await wordpressAPI.getFooter();
      setFooterData(data);
    };
    loadFooter();
  }, []);

  // ✅ Fetch Translations
  useEffect(() => {
    const loadTranslations = async () => {
      const data = await wordpressAPI.getTranslations();
      setTranslations(data);
    };
    loadTranslations();
  }, []);

  // ✅ Format WP URLs
  const formatUrl = (url: string) => {
    if (url.includes(process.env.NEXT_PUBLIC_WORDPRESS_URL!)) {
      return url.replace(process.env.NEXT_PUBLIC_WORDPRESS_URL!, '');
    }
    return url;
  };

  return (
    <footer className="bg-gray-900 text-white dark:bg-gray-800 dark:text-white mt-auto">
      <div className="container pt-16 pb-4">
        <div className="grid gap-8 grid-cols-2 lg:grid-cols-6">
          {/* ✅ Company Info */}
          <div className="col-span-2 lg:col-span-3 space-y-6">
            <div className="footer-title text-base font-semibold mb-4">
              {language === 'ar'
                ? translations?.footer_about_ar
                : translations?.footer_about_en}
            </div>
            <p
              className="text-sm text-gray-400"
              dir={language === 'ar' ? 'rtl' : 'ltr'}
            >
              {language === 'ar'
                ? footerData?.footer_about_ar
                : footerData?.footer_about_en}
            </p>
            <div className="text-white">
              <ContactInfo showInfo={false} showSocial={true} showMap={false} />
            </div>
          </div>

          {/* ✅ Footer Menu 1 */}
          <div className="col-span-1 lg:col-span-1">
            <div className="footer-title text-base font-semibold mb-4">
              {language === 'ar'
                ? translations?.footer_title_quicklinks_ar
                : translations?.footer_title_quicklinks_en}
            </div>
            <ul className="space-y-2">
              {footerMenu1.map((item) => (
                <li key={item.id}>
                  <Link
                    href={formatUrl(item.url)}
                    target={item.target}
                    className="text-gray-400 hover:text-white transition-colors"
                  >
                    {item.title}
                  </Link>
                </li>
              ))}
            </ul>
          </div>

          {/* ✅ Footer Menu 2 */}
          <div className="col-span-1 lg:col-span-1">
            <div className="footer-title text-base font-semibold mb-4">
              {language === 'ar'
                ? translations?.footer_title_resources_ar
                : translations?.footer_title_resources_en}
            </div>
            <ul className="space-y-2">
              {footerMenu2.map((item) => (
                <li key={item.id}>
                  <Link
                    href={formatUrl(item.url)}
                    target={item.target}
                    className="text-gray-400 hover:text-white transition-colors"
                  >
                    {item.title}
                  </Link>
                </li>
              ))}
            </ul>
          </div>

          {/* ✅ Footer 3 */}
          <div className="col-span-2 lg:col-span-1">
            <div className="footer-title text-base font-semibold mb-4">
              {language === 'ar'
                ? translations?.footer_title_contact_ar
                : translations?.footer_title_contact_en}
            </div>
            <div className="text-white">
              <ContactInfo showInfo={true} showSocial={false} showMap={false} />
            </div>
          </div>


        </div>

        {/* ✅ Bottom Copyright */}
        <div className="border-t border-gray-800 dark:border-gray-700 mt-8 pt-4 text-center text-gray-400">
          <p>&copy; {new Date().getFullYear()} All rights reserved.</p>
        </div>
      </div>
    </footer>
  );
}
