"use client";

import { useEffect, useState } from "react";
import wordpressAPI from "@/lib/wordpress";
import PostCard from "@/components/cards/PostCard";
import Pagination from "@/components/common/Pagination";
import SectionHeader from "@/components/common/SectionHeader";
import { useLanguage } from "@/contexts/LanguageContext";

export default function Home() {
  const { language } = useLanguage();

  // ✅ State management
  const [posts, setPosts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);

  // ✅ Translations
  const [translations, setTranslations] = useState<any | null>(null);
  const [loadingTranslations, setLoadingTranslations] = useState(true);

  // ✅ Limit (number of posts per page) controlled by .env
  const LIMIT = parseInt(process.env.NEXT_PUBLIC_POSTS_LIMIT || "6", 10);

  // ✅ Fetch posts when language OR page changes
  useEffect(() => {
    const fetchPosts = async () => {
      try {
        setLoading(true);
        const data = await wordpressAPI.getPosts(language, LIMIT, currentPage);
        setPosts(data.items || []);
        setTotalPages(data.totalPages || 1);
      } catch (error) {
        console.error("Error loading posts:", error);
        setPosts([]);
        setTotalPages(1);
      } finally {
        setLoading(false);
      }
    };
    fetchPosts();
  }, [language, currentPage]);

  // ✅ Fetch translations
  useEffect(() => {
    const loadTranslations = async () => {
      try {
        setLoadingTranslations(true);
        const data = await wordpressAPI.getTranslations();
        setTranslations(data);
      } catch (error) {
        console.error("Error loading translations:", error);
      } finally {
        setLoadingTranslations(false);
      }
    };
    loadTranslations();
  }, []);

  return (
    <div className="te-posts">
      <div className="container my-8">

        {/* ✅ Section Header */}
        <SectionHeader
          title={
            loadingTranslations
              ? "..." // Skeleton/fallback
              : language === "ar"
                ? translations?.blog_title_ar
                : translations?.blog_title_en
          }
          subtitle={
            loadingTranslations
              ? ""
              : language === "ar"
                ? translations?.blog_subtitle_ar
                : translations?.blog_subtitle_en
          }
        />

        {/* ✅ Posts Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {loading
            ? Array.from({ length: LIMIT }).map((_, i) => (
              <div
                key={i}
                className="animate-pulse bg-white dark:bg-gray-800 rounded-lg shadow-md overflow-hidden"
              >
                <div className="h-48 w-full bg-gray-200 dark:bg-gray-700" />
                <div className="p-6 space-y-4">
                  <div className="h-6 bg-gray-200 dark:bg-gray-700 rounded w-3/4" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-full" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-5/6" />
                </div>
              </div>
            ))
            : posts.map((post) => <PostCard key={post.id} post={post} />)}
        </div>

        {/* ✅ Pagination Component */}
        <Pagination
          currentPage={currentPage}
          totalPages={totalPages}
          onPageChange={(page) => setCurrentPage(page)}
        />
      </div>
    </div>
  );
}
