/**
 * Header Scroll Behavior, Mobile Menu & Search
 */
(function() {
    'use strict';

    const header = document.getElementById('main-header');
    const mobileMenuBtn = document.getElementById('mobile-menu-btn');
    const mobileMenu = document.getElementById('mobile-menu');
    const menuIconOpen = document.getElementById('menu-icon-open');
    const menuIconClose = document.getElementById('menu-icon-close');
    const searchBtn = document.getElementById('search-btn');
    const searchDropdown = document.getElementById('search-dropdown');
    const searchClose = document.getElementById('search-close');
    const searchInput = document.getElementById('search-input');

    const isHomePage = header ? header.dataset.page === 'home' : false;
    let isMenuOpen = false;
    let headerVisible = false;

    // Search Dropdown Toggle
    if (searchBtn && searchDropdown) {
        let isSearchOpen = false;

        function openSearch() {
            isSearchOpen = true;
            searchDropdown.classList.remove('hidden');
            if (searchInput) searchInput.focus();
        }

        function closeSearch() {
            isSearchOpen = false;
            searchDropdown.classList.add('hidden');
            if (searchInput) searchInput.value = '';
        }

        searchBtn.addEventListener('click', function() {
            if (isSearchOpen) {
                closeSearch();
            } else {
                openSearch();
            }
        });

        if (searchClose) {
            searchClose.addEventListener('click', closeSearch);
        }

        // Close on Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && isSearchOpen) {
                closeSearch();
            }
        });

        // Close when clicking outside
        document.addEventListener('click', function(e) {
            if (isSearchOpen && !searchDropdown.contains(e.target) && !searchBtn.contains(e.target)) {
                closeSearch();
            }
        });
    }

    // Mobile Menu Toggle
    function openMobileMenu() {
        isMenuOpen = true;
        mobileMenu.classList.remove('hidden');
        menuIconOpen.classList.add('hidden');
        menuIconClose.classList.remove('hidden');
        document.body.style.overflow = 'hidden';

        // Ensure header is visible and has white background when menu is open
        if (isHomePage) {
            header.style.position = 'fixed';
        }
        header.style.transform = 'translateY(0)';
        header.classList.add('bg-white', 'shadow-sm');
        headerVisible = true;
    }

    function closeMobileMenu() {
        isMenuOpen = false;
        mobileMenu.classList.add('hidden');
        menuIconOpen.classList.remove('hidden');
        menuIconClose.classList.add('hidden');
        document.body.style.overflow = '';

        // Restore header state based on scroll position
        if (isHomePage) {
            if (window.scrollY <= 100) {
                // At top - reset to absolute positioning
                header.style.position = 'absolute';
                header.style.transform = 'translateY(0)';
                headerVisible = false;
                header.classList.remove('bg-white', 'shadow-sm');
            } else {
                // Scrolled - keep header fixed and visible
                header.style.position = 'fixed';
                header.style.transform = 'translateY(0)';
                headerVisible = true;
                header.classList.add('bg-white', 'shadow-sm');
            }
        }
    }

    if (mobileMenuBtn && mobileMenu) {
        mobileMenuBtn.addEventListener('click', function() {
            if (isMenuOpen) {
                closeMobileMenu();
            } else {
                openMobileMenu();
            }
        });

        // Close menu when clicking a link
        mobileMenu.querySelectorAll('a').forEach(function(link) {
            link.addEventListener('click', closeMobileMenu);
        });

        // Close menu on window resize to desktop
        window.addEventListener('resize', function() {
            if (window.innerWidth >= 1024 && isMenuOpen) {
                closeMobileMenu();
            }
        });
    }

    // Header Scroll Behavior
    if (!header) return;

    let lastScrollY = window.scrollY;
    let ticking = false;

    function showHeader() {
        if (headerVisible) return;
        headerVisible = true;

        if (isHomePage) {
            header.style.position = 'fixed';
        }
        header.style.transform = 'translateY(0)';

        // Add background for visibility when scrolling (only if menu is closed)
        if (isHomePage && !isMenuOpen) {
            header.classList.add('bg-white', 'shadow-sm');
        }
    }

    function hideHeader() {
        if (!headerVisible) return;
        // Don't hide header if mobile menu is open
        if (isMenuOpen) return;
        headerVisible = false;

        header.style.transform = 'translateY(-100%)';
    }

    function resetHeader() {
        headerVisible = false;

        if (isHomePage) {
            header.style.position = 'absolute';
        }
        header.style.transform = 'translateY(0)';

        // Remove background on home page when at top (only if menu is closed)
        if (isHomePage && !isMenuOpen) {
            header.classList.remove('bg-white', 'shadow-sm');
        }
    }

    function updateHeader() {
        const currentScrollY = window.scrollY;
        const scrollDelta = currentScrollY - lastScrollY;

        if (currentScrollY <= 100) {
            // Near top of page - reset to absolute (home) or show (other pages)
            resetHeader();
        } else if (scrollDelta < 0) {
            // Scrolling UP - show header
            showHeader();
        } else if (scrollDelta > 0) {
            // Scrolling DOWN - hide header
            hideHeader();
        }

        lastScrollY = currentScrollY;
        ticking = false;
    }

    function onScroll() {
        if (!ticking) {
            window.requestAnimationFrame(updateHeader);
            ticking = true;
        }
    }

    // Set initial styles for non-home pages
    if (!isHomePage) {
        header.classList.add('bg-white', 'shadow-sm');
    }

    window.addEventListener('scroll', onScroll, { passive: true });
})();
