/**
 * Universal Filter System
 * Works for ads.php, companies.php, and funds.php
 */

document.addEventListener('DOMContentLoaded', function() {
    // Check if we're on a page with filters
    const searchInput = document.getElementById('search-ads') ||
                       document.getElementById('search-companies') ||
                       document.getElementById('search-funds');

    if (!searchInput) return; // Exit if no search input found

    // Get elements based on page type
    const filterTabs = document.querySelectorAll('.filter-tab');
    const grid = document.getElementById('ads-grid') ||
                 document.getElementById('companies-grid') ||
                 document.getElementById('funds-grid');

    const sectorFilter = document.getElementById('sector-filter');
    const priceFilter = document.getElementById('price-filter');
    const sortFilter = document.getElementById('sort-filter');
    const typeFilter = document.getElementById('type-filter');
    const riskFilter = document.getElementById('risk-filter');
    const statusFilter = document.getElementById('status-filter');
    const resetButton = document.getElementById('reset-filters');
    const resultsCount = document.getElementById('results-count');

    let currentFilter = 'all';

    // Function to apply all filters
    function applyFilters() {
        if (!grid) return;

        const searchTerm = searchInput.value.trim().toLowerCase();
        const sectorValue = sectorFilter ? sectorFilter.value : '';
        const priceValue = priceFilter ? priceFilter.value : '';
        const sortValue = sortFilter ? sortFilter.value : '';
        const typeValue = typeFilter ? typeFilter.value : '';
        const riskValue = riskFilter ? riskFilter.value : '';
        const statusValue = statusFilter ? statusFilter.value : '';

        // Get cards - different selector based on page
        let cards = Array.from(grid.querySelectorAll('.ad-card, .company-card, .fund-card')); // For all card types
        if (cards.length === 0) {
            cards = Array.from(grid.querySelectorAll('div[data-name]')); // Fallback for divs with data-name
        }
        if (cards.length === 0) {
            cards = Array.from(grid.querySelectorAll('a[data-type]')); // Fallback for links
        }

        let visibleCount = 0;

        cards.forEach(card => {
            // Get card data
            const cardType = card.getAttribute('data-type');
            const cardName = (card.getAttribute('data-name') || card.querySelector('h3')?.textContent || '').toLowerCase();

            const cardSectorElement = card.querySelector('h3 + p');
            const cardSector = cardSectorElement ? cardSectorElement.textContent.trim() : '';

            const cardPriceElement = card.querySelector('.text-xl.font-bold, .text-xs.font-bold');
            const cardPrice = cardPriceElement ? parseInt(cardPriceElement.textContent) : 0;

            // For funds - get data attributes
            const cardFundType = card.getAttribute('data-fund-type') || '';
            const cardRisk = card.getAttribute('data-risk') || '';
            const cardBadge = card.getAttribute('data-badge') || '';

            // Check tab filter (for ads with buy/sell tabs)
            let tabMatches = true;
            if (currentFilter !== 'all' && cardType) {
                tabMatches = cardType === currentFilter;
            }

            // Check search filter
            const searchMatches = searchTerm === '' ||
                cardName.includes(searchTerm) ||
                cardSector.toLowerCase().includes(searchTerm);

            // Check sector/type filter
            let sectorMatches = true;
            if (sectorValue) {
                sectorMatches = cardSector === sectorValue;
            }

            // Check fund type filter
            let typeMatches = true;
            if (typeValue) {
                typeMatches = cardFundType === typeValue;
            }

            // Check price filter
            let priceMatches = true;
            if (priceValue === '0-100') {
                priceMatches = cardPrice >= 0 && cardPrice <= 100;
            } else if (priceValue === '100-200') {
                priceMatches = cardPrice > 100 && cardPrice <= 200;
            } else if (priceValue === '200-500') {
                priceMatches = cardPrice > 200 && cardPrice <= 500;
            } else if (priceValue === '500+') {
                priceMatches = cardPrice > 500;
            }

            // Check risk filter (for funds)
            let riskMatches = true;
            if (riskValue) {
                riskMatches = cardRisk === riskValue;
            }

            // Check status/badge filter (for funds)
            let statusMatches = true;
            if (statusValue) {
                statusMatches = cardBadge === statusValue;
            }

            // Show card only if all filters match
            if (tabMatches && searchMatches && sectorMatches && typeMatches && priceMatches && riskMatches && statusMatches) {
                card.style.display = '';
                card.dataset.visible = 'true';
                visibleCount++;
            } else {
                card.style.display = 'none';
                card.dataset.visible = 'false';
            }
        });

        // Apply sorting
        if (sortValue) {
            const visibleCards = cards.filter(card => card.dataset.visible === 'true');

            visibleCards.sort((a, b) => {
                const priceA = parseInt(a.querySelector('.text-xl.font-bold, .text-xs.font-bold')?.textContent || '0');
                const priceB = parseInt(b.querySelector('.text-xl.font-bold, .text-xs.font-bold')?.textContent || '0');

                if (sortValue === 'price-asc') {
                    return priceA - priceB;
                } else if (sortValue === 'price-desc') {
                    return priceB - priceA;
                } else if (sortValue === 'ads-desc') {
                    const adsA = parseInt(a.querySelector('.bg-secondary-50')?.textContent || '0');
                    const adsB = parseInt(b.querySelector('.bg-secondary-50')?.textContent || '0');
                    return adsB - adsA;
                }
                return 0;
            });

            // Reorder visible cards in DOM
            visibleCards.forEach(card => {
                grid.appendChild(card);
            });
        }

        // Update results count
        if (resultsCount) {
            const total = cards.length;
            const pageType = document.getElementById('ads-grid') ? 'إعلان' :
                           document.getElementById('companies-grid') ? 'شركة' : 'صناديق';

            if (pageType === 'إعلان') {
                resultsCount.textContent = `${visibleCount} ${pageType} متاح`;
            } else {
                resultsCount.textContent = `عرض النتائج 1 إلى ${visibleCount} من ${total} إجمالاً ${pageType}`;
            }
        }
    }

    // Tab filter click handler (for ads page)
    if (filterTabs.length > 0) {
        filterTabs.forEach(tab => {
            tab.addEventListener('click', function() {
                // Remove active class from all tabs
                filterTabs.forEach(t => {
                    t.classList.remove('active', 'bg-primary-500', 'text-white');
                    t.classList.add('bg-gray-50', 'text-gray-800');
                });

                // Add active class to clicked tab
                this.classList.add('active', 'bg-primary-500', 'text-white');
                this.classList.remove('bg-gray-50', 'text-gray-800');

                currentFilter = this.dataset.filter;
                applyFilters();
            });
        });
    }

    // Search input handler
    if (searchInput) {
        searchInput.addEventListener('input', applyFilters);
    }

    // Dropdown filters handlers
    if (sectorFilter) sectorFilter.addEventListener('change', applyFilters);
    if (priceFilter) priceFilter.addEventListener('change', applyFilters);
    if (sortFilter) sortFilter.addEventListener('change', applyFilters);
    if (typeFilter) typeFilter.addEventListener('change', applyFilters);
    if (riskFilter) riskFilter.addEventListener('change', applyFilters);
    if (statusFilter) statusFilter.addEventListener('change', applyFilters);

    // Reset button handler
    if (resetButton) {
        resetButton.addEventListener('click', function() {
            // Reset search input
            searchInput.value = '';

            // Reset all dropdowns
            if (sectorFilter) sectorFilter.value = '';
            if (priceFilter) priceFilter.value = '';
            if (sortFilter) sortFilter.value = '';
            if (typeFilter) typeFilter.value = '';
            if (riskFilter) riskFilter.value = '';
            if (statusFilter) statusFilter.value = '';

            // Reset to "All" tab (for ads page)
            if (filterTabs.length > 0) {
                filterTabs.forEach(t => {
                    t.classList.remove('active', 'bg-primary-500', 'text-white');
                    t.classList.add('bg-gray-50', 'text-gray-800');
                });
                filterTabs[0].classList.add('active', 'bg-primary-500', 'text-white');
                filterTabs[0].classList.remove('bg-gray-50', 'text-gray-800');
                currentFilter = 'all';
            }

            // Apply filters to show all
            applyFilters();
        });
    }

    // Initialize count
    applyFilters();
});
